//
//  B5Database.h
//  ApplicationFrameworks
//
//  Created by Rick Fillion on 2015-03-18.
//
//

#import <Foundation/Foundation.h>
#import <OnePasswordDataModel/OnePasswordDataModel.h>
#import "OPDatabase.h"

NS_ASSUME_NONNULL_BEGIN

@class B5UserAccount;
@class B5VaultAccess;

extern const NSString *B5DatabaseSchemaVersionMax;

@interface B5Database : OPDatabase <OPDatabaseBasicItemSupport,
		OPDatabaseProfileSupport,
		OPDatabaseCategoriesSupport,
		OPDatabaseTransactionIdentificationSupport,
		OPDatabaseOptionalB5AccountSupport,
		OPDatabaseOptionalB5SyncSupport>

@property (nonatomic, readonly) NSURL * databaseURL;

@property (nonatomic, readonly) NSString * databaseUUID;

@property (nonatomic, readonly, copy) NSArray *allUserAccounts;
@property (nonatomic, readonly) B5UserAccount *masterUserAccount;
// secondaryMasterUserAccount Returns what would be the master user account if you were to delete the master user account. You need to know this before you delete the masterUserAccount to migrate login dictionaries
@property (nonatomic, readonly) B5UserAccount *secondaryMasterUserAccount;

@property (nonatomic, strong, nullable) NSString * txSyncKeychainIdentifier; // Both identifies the currently running syncer and the fact that this transaction is performed during sync

+ (B5Database *)activeDatabase;
+ (void)setActiveDatabase:(nullable B5Database *)database;
+ (void)destroyActiveDatabase;

- (id)initWithURL:(NSURL *)URL;

/**
 * Synchronous method used in unit tests.
 */
- (BOOL)openDatabaseWithError:(NSError **)error;

/**
 * The offline operation will be performed when the SQLite file is closed. The SQLite file will be reopen and caches cleared after the operation is completed.
 * If before: block is not nil and returns NO then the offline operation is not performed.
 */
- (OPDatabaseOperation *)performOfflineOperation:(BOOL (^)(B5Database *db, NSError **error))offlineOperation before:(nullable BOOL (^)(B5Database *db, BOOL *rollback))before after:(nullable void (^)(BOOL success, NSError *error))after;

- (void)performSyncTransaction:(void (^)(B5Database *db, BOOL *rollback))transaction;
- (void)performAsyncTransaction:(void (^)(B5Database *db, BOOL *rollback))transaction completion:(nullable void (^)(BOOL success))completion;
- (void)performAsyncTransaction:(void (^)(B5Database *db, BOOL *rollback))transaction;

- (void)performLowPriorityAsyncTransaction:(void (^)(B5Database *db, BOOL *rollback))transaction completion:(void (^)(BOOL success))completion;
- (void)performLowPriorityAsyncTransaction:(void (^)(B5Database *db, BOOL *rollback))transaction;
- (void)performLowPrioritySyncTransaction:(void (^)(B5Database *db, BOOL *rollback))transaction;

- (NSArray *)allProfiles;
- (NSArray *)allUnlockedProfiles;
- (NSString *)uniqueProfileNameForName:(NSString *)profileName;

//
// The txSelect and txUpdate methods below must always be called within the performSyncTransaction or performAsyncTransaction
//

- (BOOL)unlockProfile:(B5Profile *)profile withPassword:(NSString *)password error:(NSError **)error;

- (B5Profile *)profileWithId:(OPID)profileId;
- (B5Profile *)profileWithUUID:(NSString *)profileUUID;
- (B5Profile *)profileWithName:(NSString *)profileName;

- (NSArray *)txSelectOldTombstonesForProfile:(B5Profile *)profile;

- (BOOL)txDeleteCategoryWithUUID:(NSString *)UUID accountId:(OPID)accountId error:(NSError **)error;

//
// Utility methods
//
- (BOOL)txClearAutosaveTokensWithError:(NSError **)error;

- (NSString *)txSelectConfigValueForKey:(NSString *)key;
- (BOOL)txSetConfigValue:(NSString *)value forKey:(NSString *)key;

/// Must not be run within a transaction
- (BOOL)vacuum:(NSError **)error;

/// You don't want to use these, trust me
- (id)_txSelectItemWithId:(OPID)oid loadCompletely:(BOOL)loadCompletely assertIfNotFound:(BOOL)assertIfNotFound;

@end

NS_ASSUME_NONNULL_END